<?php
/**
 * Visitor Counter Extension for PriviMetrics
 * Official extension by WebOrbiton Team
 * Counts total visits across all pages and displays a dashboard widget
 */

$dataFile = __DIR__ . '/data.json';

// --- Helper functions with proper file locking ---
function getDataWithLock($key, $dataFile) {
    if (!file_exists($dataFile)) return null;

    $fp = fopen($dataFile, 'r');
    if (!$fp) return null;

    flock($fp, LOCK_SH); // shared lock for reading
    $contents = '';
    if (filesize($dataFile) > 0) {
        $contents = fread($fp, filesize($dataFile));
    }
    $data = $contents ? json_decode($contents, true) : [];
    flock($fp, LOCK_UN);
    fclose($fp);

    return $data[$key] ?? null;
}

function setDataWithLock($key, $value, $dataFile) {
    $data = [];
    $fp = fopen($dataFile, file_exists($dataFile) ? 'c+' : 'w');
    if (!$fp) return false;

    flock($fp, LOCK_EX);

    if (file_exists($dataFile) && filesize($dataFile) > 0) {
        rewind($fp);
        $contents = fread($fp, filesize($dataFile));
        $data = $contents ? json_decode($contents, true) : [];
    }

    $data[$key] = $value;

    ftruncate($fp, 0);
    rewind($fp);
    fwrite($fp, json_encode($data, JSON_PRETTY_PRINT));
    fflush($fp);
    flock($fp, LOCK_UN);
    fclose($fp);

    return true;
}

// --- Initialize persistent data ---
$totalVisits = getDataWithLock('total_visits', $dataFile);
if ($totalVisits === null) {
    setDataWithLock('total_visits', 0, $dataFile);
    setDataWithLock('install_date', date('Y-m-d H:i:s'), $dataFile);
}

// --- Hook: Increment visit count after each analytics save ---
$ext['registerHook']('after_analytics_save', function($data) use ($dataFile) {
    $current = getDataWithLock('total_visits', $dataFile) ?? 0;
    setDataWithLock('total_visits', $current + 1, $dataFile);
    return $data;
});

// --- Hook: Add dashboard widget ---
$ext['registerHook']('custom_dashboard_widget', function($html) use ($dataFile) {
    $total = getDataWithLock('total_visits', $dataFile) ?? 0;
    $installDate = getDataWithLock('install_date', $dataFile) ?? 'Unknown';

    $widget = '
    <div class="stat-card" style="background: linear-gradient(135deg, #ea6666 0%, #764ba2 100%); color: white;">
        <div class="stat-label" style="color: rgba(255,255,255,0.9);">Total Visits</div>
        <div class="stat-value">' . number_format($total) . '</div>
        <div style="font-size: 11px; margin-top: 8px; opacity: 0.8;">
            Since ' . date('M d, Y', strtotime($installDate)) . '
        </div>
    </div>
    ';

    return $html . $widget;
});
