<?php
/**
 * Plugin Name: Official PriviMetrics for WordPress
 * Plugin URI: https://marketplace.privimetrics.com/product.php?id=privimetrics-for-wordpress
 * Description: Official PriviMetrics analytics integration for WordPress. Privacy-focused, GDPR-compliant analytics without cookies.
 * Version: 1.5.5
 * Author: PriviMetrics
 * Author URI: https://privimetrics.com
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: privimetrics
 * Domain Path: /languages
 * Requires at least: 5.0
 * Requires PHP: 7.2
 */

if (!defined('ABSPATH')) {
    exit;
}

define('PRIVIMETRICS_VERSION', '1.5.5');
define('PRIVIMETRICS_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('PRIVIMETRICS_PLUGIN_URL', plugin_dir_url(__FILE__));

class PriviMetrics {
    
    private static $instance = null;
    
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'register_settings'));
        add_action('wp_enqueue_scripts', array($this, 'enqueue_tracking_script'));
        add_filter('plugin_action_links_' . plugin_basename(__FILE__), array($this, 'add_action_links'));
        add_filter('script_loader_tag', array($this, 'add_script_attributes'), 10, 2);
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));
    }
    
    public function activate() {
        if (!get_option('privimetrics_options')) {
            add_option('privimetrics_options', array(
                'site_id' => '',
                'enabled' => true,
                'track_admin' => false,
                'script_url' => 'https://yourdomain.com/privimetrics-div.js'
            ));
        }
    }
    
    public function deactivate() {

    }
    
    public function add_admin_menu() {
        add_options_page(
            __('PriviMetrics Settings', 'privimetrics'),
            __('PriviMetrics', 'privimetrics'),
            'manage_options',
            'privimetrics',
            array($this, 'settings_page')
        );
    }
    
    public function register_settings() {
        register_setting('privimetrics_options_group', 'privimetrics_options', array($this, 'sanitize_options'));
    }
    
    public function sanitize_options($input) {
        $sanitized = array();
        
        if (isset($input['site_id'])) {
            $sanitized['site_id'] = sanitize_text_field($input['site_id']);
        }
        
        $sanitized['enabled'] = isset($input['enabled']) ? (bool) $input['enabled'] : false;
        $sanitized['track_admin'] = isset($input['track_admin']) ? (bool) $input['track_admin'] : false;
        
        if (isset($input['script_url'])) {
            $sanitized['script_url'] = esc_url_raw($input['script_url']);
        }
        
        return $sanitized;
    }
    
    public function settings_page() {
        if (!current_user_can('manage_options')) {
            return;
        }
        
        $options = get_option('privimetrics_options', array(
            'site_id' => '',
            'enabled' => true,
            'track_admin' => false,
            'script_url' => 'https://yourdomain.com/privimetrics-div.js'
        ));
        
        ?>
        <div class="wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
            
            <form method="post" action="options.php">
                <?php
                settings_fields('privimetrics_options_group');
                ?>
                
                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row">
                            <label for="site_id"><?php _e('Site ID', 'privimetrics'); ?></label>
                        </th>
                        <td>
                            <input type="text" 
                                   id="site_id" 
                                   name="privimetrics_options[site_id]" 
                                   value="<?php echo esc_attr($options['site_id']); ?>" 
                                   class="regular-text" 
                                   required>
                            <p class="description">
                                <?php _e('Enter your PriviMetrics Site ID. You can find this in your PriviMetrics dashboard.', 'privimetrics'); ?>
                            </p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <label for="script_url"><?php _e('Script URL', 'privimetrics'); ?></label>
                        </th>
                        <td>
                            <input type="url" 
                                   id="script_url" 
                                   name="privimetrics_options[script_url]" 
                                   value="<?php echo esc_attr($options['script_url']); ?>" 
                                   class="regular-text">
                            <p class="description">
                                <?php _e('Tracking script URL.', 'privimetrics'); ?>
                            </p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row"><?php _e('Enable Tracking', 'privimetrics'); ?></th>
                        <td>
                            <fieldset>
                                <label for="enabled">
                                    <input type="checkbox" 
                                           id="enabled" 
                                           name="privimetrics_options[enabled]" 
                                           value="1" 
                                           <?php checked($options['enabled'], true); ?>>
                                    <?php _e('Enable PriviMetrics tracking on your website', 'privimetrics'); ?>
                                </label>
                            </fieldset>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row"><?php _e('Track Administrators', 'privimetrics'); ?></th>
                        <td>
                            <fieldset>
                                <label for="track_admin">
                                    <input type="checkbox" 
                                           id="track_admin" 
                                           name="privimetrics_options[track_admin]" 
                                           value="1" 
                                           <?php checked($options['track_admin'], true); ?>>
                                    <?php _e('Track administrator visits (not recommended)', 'privimetrics'); ?>
                                </label>
                                <p class="description">
                                    <?php _e('When disabled, visits from logged-in administrators will not be tracked.', 'privimetrics'); ?>
                                </p>
                            </fieldset>
                        </td>
                    </tr>
                </table>
                
                <?php submit_button(__('Save Settings', 'privimetrics')); ?>
            </form>
            
            <hr>
            
            <div class="privimetrics-info">
                <h2><?php _e('About PriviMetrics', 'privimetrics'); ?></h2>
                <p><?php _e('PriviMetrics is a privacy-focused analytics platform that respects user privacy while providing valuable insights.', 'privimetrics'); ?></p>
                
                <h3><?php _e('Features', 'privimetrics'); ?></h3>
                <ul style="list-style: disc; margin-left: 20px;">
                    <li><?php _e('No cookies required', 'privimetrics'); ?></li>
                    <li><?php _e('GDPR compliant', 'privimetrics'); ?></li>
                    <li><?php _e('Lightweight tracking script', 'privimetrics'); ?></li>
                    <li><?php _e('Real-time analytics', 'privimetrics'); ?></li>
                    <li><?php _e('Privacy-focused approach', 'privimetrics'); ?></li>
                </ul>
                
                <p>
                    <a href="https://privimetrics.com" target="_blank" class="button button-secondary">
                        <?php _e('Visit PriviMetrics', 'privimetrics'); ?>
                    </a>
                </p>
            </div>
        </div>
        
        <style>
            .privimetrics-info {
                max-width: 800px;
                padding: 20px;
                background: #fff;
                border: 1px solid #ccd0d4;
                border-radius: 4px;
                margin-top: 20px;
            }
            .privimetrics-info h2 {
                margin-top: 0;
            }
            .privimetrics-info ul {
                line-height: 1.8;
            }
        </style>
        <?php
    }
    
    public function enqueue_tracking_script() {
        if (is_admin()) {
            return;
        }
        
        $options = get_option('privimetrics_options');
        
        if (!$options || empty($options['site_id']) || !$options['enabled']) {
            return;
        }
        
        if (!$options['track_admin'] && is_user_logged_in() && current_user_can('manage_options')) {
            return;
        }
        
        $script_url = !empty($options['script_url']) ? $options['script_url'] : 'https://yourdomain.com/privimetrics-div.js';
        
        wp_enqueue_script(
            'privimetrics-tracker',
            $script_url,
            array(),
            PRIVIMETRICS_VERSION,
            false
        );
        
        wp_script_add_data('privimetrics-tracker', 'data-privimetrics-code', esc_attr($options['site_id']));
    }
    
    public function add_script_attributes($tag, $handle) {
        if ('privimetrics-tracker' !== $handle) {
            return $tag;
        }
        
        $options = get_option('privimetrics_options');
        $site_id = esc_attr($options['site_id']);
        
        $tag = str_replace('<script ', '<script data-privimetrics-code="' . $site_id . '" defer ', $tag);
        
        $helper_script = "
        <script>
        (function() {
            var lastUrl = location.href;
            var checkUrlChange = function() {
                if (location.href !== lastUrl) {
                    lastUrl = location.href;
                    if (window.PriviMetrics && typeof window.PriviMetrics.track === 'function') {
                        window.PriviMetrics.track('pageview');
                    }
                }
            };
            
            var originalPushState = history.pushState;
            history.pushState = function() {
                originalPushState.apply(this, arguments);
                checkUrlChange();
            };
            
            var originalReplaceState = history.replaceState;
            history.replaceState = function() {
                originalReplaceState.apply(this, arguments);
                checkUrlChange();
            };
            
            window.addEventListener('popstate', checkUrlChange);
            
            setInterval(checkUrlChange, 500);
        })();
        </script>
        ";
        
        return $tag . $helper_script;
    }
    
    public function add_action_links($links) {
        $settings_link = '<a href="' . admin_url('options-general.php?page=privimetrics') . '">' . __('Settings', 'privimetrics') . '</a>';
        array_unshift($links, $settings_link);
        return $links;
    }
}

function privimetrics() {
    return PriviMetrics::get_instance();
}

privimetrics();